#!/bin/sh
# zbx-agent2-bootstrap.sh
# Installs and configures Zabbix Agent2 on Debian/Ubuntu with ACTIVE checks (no inbound port needed).
# Usage:
#   sudo sh zbx-agent2-bootstrap.sh --server 192.168.1.10 [--branch 7.0] [--hostname myvm] [--metadata "linux,docker"] [--docker]
set -eu

SERVER=""
BRANCH="7.0"
HOSTNAME_OVERRIDE=""
METADATA="linux"
ENABLE_DOCKER=0

die() { echo "ERROR: $*" >&2; exit 1; }

while [ $# -gt 0 ]; do
  case "$1" in
    --server)   SERVER="${2:-}"; shift 2 ;;
    --branch)   BRANCH="${2:-}"; shift 2 ;;
    --hostname) HOSTNAME_OVERRIDE="${2:-}"; shift 2 ;;
    --metadata) METADATA="${2:-}"; shift 2 ;;
    --docker)   ENABLE_DOCKER=1; shift 1 ;;
    -h|--help)
      cat <<EOF
Usage: sudo sh $0 --server <zabbix_server_ip_or_dns> [options]
Options:
  --branch   <7.0|6.0|6.4>      Zabbix repo branch (default: 7.0)
  --hostname <name>             Hostname shown in Zabbix (default: system hostname)
  --metadata <string>           HostMetadata for auto-registration (default: linux)
  --docker                       Add user 'zabbix' to docker group (optional)
EOF
      exit 0
      ;;
    *) die "Unknown argument: $1" ;;
  esac
done

[ -n "$SERVER" ] || die "--server is required"

# Must be root
if [ "$(id -u)" != "0" ]; then
  die "Run as root (use sudo)"
fi

# Detect OS
[ -r /etc/os-release ] || die "/etc/os-release not found"
# shellcheck disable=SC1091
. /etc/os-release

CODENAME="${VERSION_CODENAME:-}"
[ -n "$CODENAME" ] || die "VERSION_CODENAME is empty (unsupported OS?)"

case "${ID:-}" in
  debian) REPO_BASE="https://repo.zabbix.com/zabbix/${BRANCH}/debian" ;;
  ubuntu) REPO_BASE="https://repo.zabbix.com/zabbix/${BRANCH}/ubuntu" ;;
  *)
    echo "${ID_LIKE:-}" | grep -qi debian || die "Only Debian/Ubuntu are supported (ID=${ID:-unknown})"
    REPO_BASE="https://repo.zabbix.com/zabbix/${BRANCH}/debian"
    ;;
esac

# Determine hostname for Zabbix
if [ -n "$HOSTNAME_OVERRIDE" ]; then
  ZBX_HOSTNAME="$HOSTNAME_OVERRIDE"
else
  ZBX_HOSTNAME="$(hostname -s 2>/dev/null || hostname || echo "unknown-host")"
fi

echo "[*] Installing prerequisites..."
export DEBIAN_FRONTEND=noninteractive
apt-get update -y
apt-get install -y --no-install-recommends ca-certificates curl gnupg

echo "[*] Adding Zabbix repo key..."
install -d -m 0755 /usr/share/keyrings
curl -fsSL "https://repo.zabbix.com/zabbix-official-repo.key" | gpg --dearmor -o /usr/share/keyrings/zabbix.gpg

echo "[*] Adding Zabbix repo list..."
cat >/etc/apt/sources.list.d/zabbix.list <<EOF
deb [signed-by=/usr/share/keyrings/zabbix.gpg] ${REPO_BASE} ${CODENAME} main
EOF

apt-get update -y

echo "[*] Installing zabbix-agent2..."
apt-get install -y zabbix-agent2

echo "[*] Writing Agent2 config (active-only)..."
install -d -m 0755 /etc/zabbix/zabbix_agent2.d
cat >/etc/zabbix/zabbix_agent2.d/99-bootstrap.conf <<EOF
### Managed by zbx-agent2-bootstrap.sh
Server=${SERVER}
ServerActive=${SERVER}
Hostname=${ZBX_HOSTNAME}
HostMetadata=${METADATA}

# Active-only: do not listen on TCP port for passive checks
ListenPort=0
EOF

echo "[*] Enabling and restarting zabbix-agent2..."
systemctl enable --now zabbix-agent2
systemctl restart zabbix-agent2

if [ "$ENABLE_DOCKER" -eq 1 ]; then
  if getent group docker >/dev/null 2>&1; then
    echo "[*] Adding user 'zabbix' to docker group..."
    usermod -aG docker zabbix || true
    echo "[!] NOTE: docker.sock access is powerful. Prefer docker-socket-proxy if you want safer Docker monitoring."
  else
    echo "[!] docker group not found; skipping --docker"
  fi
fi

echo "[*] Agent status (first lines):"
systemctl --no-pager --full status zabbix-agent2 | sed -n '1,12p'

echo
echo "[OK] Installed and configured Zabbix Agent2 (active-only)."
echo "    Hostname:      ${ZBX_HOSTNAME}"
echo "    ServerActive:  ${SERVER}"
echo "    HostMetadata:  ${METADATA}"
echo
echo "Next step on Zabbix Server: create an Autoregistration Action that matches HostMetadata='${METADATA}'"
echo "and links the appropriate template (e.g., 'Linux by Zabbix agent active')."
